﻿using Org.BouncyCastle.Pkcs;
using Org.BouncyCastle.Security;
using Org.BouncyCastle.X509;
using System;
using System.Collections;
using System.IO;
using System.Linq;
using Org.BouncyCastle.Crypto.Tls;


namespace Security
{
    static class BouncyCastleHelpers
    {
        public static DefaultTlsSignerCredentials Credentials(TlsContext Context, byte[] PfxFile, string Password)
        {
            Pkcs12Store pfx = new Pkcs12Store(new MemoryStream(PfxFile), Password.ToCharArray());
            string alias = (from string a in pfx.Aliases where pfx.GetCertificateChain(a) != null select a).First();
            var ClientCertificate = new Certificate(Array.ConvertAll(pfx.GetCertificateChain(alias), (x) => x.Certificate.CertificateStructure));
            var ClientPrivateKey = pfx.GetKey(alias).Key;
            return new DefaultTlsSignerCredentials(Context, ClientCertificate, ClientPrivateKey);
        }
    }

    class MinimalBouncyCastleClient : DefaultTlsClient, TlsAuthentication
    {
        public Stream TlsStream(Stream OuterStream, string Address, int Port)
        {
            TlsClientProtocol handler = new TlsClientProtocol(OuterStream, new SecureRandom());
            handler.Connect(this);
            return handler.Stream;
        }

        public override TlsAuthentication GetAuthentication() { return this; }

        public TlsCredentials GetClientCredentials(CertificateRequest certificateRequest)
        {
            return BouncyCastleHelpers.Credentials(mContext, Properties.Resources.client, "password");
        }

        public void NotifyServerCertificate(Certificate serverCertificate)
        {
            DicomObjects.DicomGlobal.Log(serverCertificate.GetCertificateAt(0).Subject + " (BC)");
        }
    }

    class MinimalBouncyCastleServer : DefaultTlsServer
    {
        public static Stream GetDecryptedStream(Stream InnerStream, string Address, int Port)
        {
            // this is done to ensure that each connection gets its "own" TlsServer, as interactions can otherwise occur
            return new MinimalBouncyCastleServer().TlsStream(InnerStream, Address, Port);
        }

        public Stream TlsStream(Stream InnerStream, string Address, int Port)
        {
            TlsServerProtocol ph = new TlsServerProtocol(InnerStream, new SecureRandom());
            ph.Accept(this);
            return ph.Stream;
        }
        protected override TlsSignerCredentials GetRsaSignerCredentials()
        {
            return BouncyCastleHelpers.Credentials(mContext, Properties.Resources.server, "password");
        }

        public override CertificateRequest GetCertificateRequest()
        {
            byte[] certificateTypes = new byte[]{ ClientCertificateType.rsa_sign,
                ClientCertificateType.dss_sign, ClientCertificateType.ecdsa_sign };

            IList certificateAuthorities = new ArrayList();
            certificateAuthorities.Add(new X509CertificateParser().ReadCertificate(Properties.Resources.CARoot).SubjectDN);
            return new CertificateRequest(certificateTypes, null, certificateAuthorities);
        }

        public override void NotifyClientCertificate(Certificate clientCertificate)
        {
            if(!clientCertificate.IsEmpty)
                DicomObjects.DicomGlobal.Log(clientCertificate.GetCertificateAt(0).Subject + " (BouncyCastle)");
        }
    }
}
